import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { BookOpen, User, TrendingUp, Award, Target, Zap, BarChart3, CheckCircle2, Clock, Calendar, ArrowUpRight, Sparkles } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyCourses.css'

const MyCourses = () => {
  const [courses, setCourses] = useState([])

  useEffect(() => {
    const saved = localStorage.getItem('egeCourses')
    if (saved) {
      try {
        setCourses(JSON.parse(saved))
      } catch (e) {
        setCourses([])
      }
    } else {
      const defaultCourses = [
        {
          id: 1,
          subject: 'Математика',
          instructor: 'Александр Петров',
          progress: 75,
          testResults: [
            { name: 'Пробный ЕГЭ №1', score: 82, maxScore: 100 },
            { name: 'Пробный ЕГЭ №2', score: 88, maxScore: 100 }
          ]
        },
        {
          id: 2,
          subject: 'Русский язык',
          instructor: 'Елена Соколова',
          progress: 60,
          testResults: [
            { name: 'Пробный ЕГЭ №1', score: 90, maxScore: 100 }
          ]
        },
        {
          id: 3,
          subject: 'Физика',
          instructor: 'Дмитрий Новиков',
          progress: 45,
          testResults: []
        }
      ]
      setCourses(defaultCourses)
      localStorage.setItem('egeCourses', JSON.stringify(defaultCourses))
    }
  }, [])

  const totalProgress = courses.length > 0
    ? Math.round(courses.reduce((sum, c) => sum + c.progress, 0) / courses.length)
    : 0

  const averageScore = courses.length > 0
    ? Math.round(
        courses
          .flatMap(c => c.testResults)
          .reduce((sum, t) => sum + (t.score / t.maxScore) * 100, 0) /
        courses.flatMap(c => c.testResults).length
      )
    : 0

  return (
    <div className="my-courses-dashboard">
      <div className="container">
        <motion.div 
          className="dashboard-header"
          initial={{ opacity: 0, y: -20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <div className="header-content-wrapper">
            <div>
              <h1 className="dashboard-title">Добро пожаловать!</h1>
              <p className="dashboard-subtitle">Ваша панель управления курсами</p>
            </div>
            <div className="header-badge">
              <Sparkles size={18} strokeWidth={2.5} />
              <span>Активный ученик</span>
            </div>
          </div>
        </motion.div>

        <div className="dashboard-grid">
          <motion.div 
            className="widget widget-large widget-primary"
            initial={{ opacity: 0, scale: 0.95 }}
            animate={{ opacity: 1, scale: 1 }}
            transition={{ delay: 0.1 }}
            whileHover={{ scale: 1.02 }}
          >
            <div className="widget-header">
              <div className="widget-icon-large">
                <TrendingUp size={32} strokeWidth={2.5} />
              </div>
              <div className="widget-content">
                <div className="widget-label">Общий прогресс</div>
                <div className="widget-value-large">{totalProgress}%</div>
              </div>
            </div>
            <div className="progress-ring-container">
              <svg className="progress-ring" viewBox="0 0 120 120">
                <circle className="progress-ring-bg" cx="60" cy="60" r="54" />
                <motion.circle
                  className="progress-ring-fill"
                  cx="60"
                  cy="60"
                  r="54"
                  initial={{ strokeDasharray: 0 }}
                  animate={{ strokeDasharray: `${(totalProgress / 100) * 339.3}, 339.3` }}
                  transition={{ duration: 1, delay: 0.3 }}
                />
              </svg>
            </div>
          </motion.div>

          <motion.div 
            className="widget widget-medium"
            initial={{ opacity: 0, scale: 0.95 }}
            animate={{ opacity: 1, scale: 1 }}
            transition={{ delay: 0.2 }}
            whileHover={{ y: -5 }}
          >
            <div className="widget-icon">
              <BookOpen size={24} strokeWidth={2.5} />
            </div>
            <div className="widget-content">
              <div className="widget-value">{courses.length}</div>
              <div className="widget-label">Активных курсов</div>
            </div>
            <ArrowUpRight size={20} strokeWidth={2} className="widget-arrow" />
          </motion.div>

          <motion.div 
            className="widget widget-medium"
            initial={{ opacity: 0, scale: 0.95 }}
            animate={{ opacity: 1, scale: 1 }}
            transition={{ delay: 0.3 }}
            whileHover={{ y: -5 }}
          >
            <div className="widget-icon">
              <Award size={24} strokeWidth={2.5} />
            </div>
            <div className="widget-content">
              <div className="widget-value">{averageScore}</div>
              <div className="widget-label">Средний балл</div>
            </div>
            <ArrowUpRight size={20} strokeWidth={2} className="widget-arrow" />
          </motion.div>

          <motion.div 
            className="widget widget-medium"
            initial={{ opacity: 0, scale: 0.95 }}
            animate={{ opacity: 1, scale: 1 }}
            transition={{ delay: 0.4 }}
            whileHover={{ y: -5 }}
          >
            <div className="widget-icon">
              <Target size={24} strokeWidth={2.5} />
            </div>
            <div className="widget-content">
              <div className="widget-value">
                {courses.flatMap(c => c.testResults).length}
              </div>
              <div className="widget-label">Пройдено тестов</div>
            </div>
            <ArrowUpRight size={20} strokeWidth={2} className="widget-arrow" />
          </motion.div>
        </div>

        <ImageSlider />

        <motion.div 
          className="courses-widget"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ delay: 0.5 }}
        >
          <div className="widget-section-header">
            <h2 className="section-title-modern">Мои курсы</h2>
            <div className="section-actions">
              <span className="courses-count">{courses.length} курсов</span>
            </div>
          </div>
          {courses.length === 0 ? (
            <div className="empty-widget">
              <div className="empty-icon-modern">
                <BookOpen size={48} strokeWidth={2} />
              </div>
              <h3 className="empty-title-modern">Начните обучение</h3>
              <p className="empty-text-modern">Запишитесь на первый курс подготовки к ЕГЭ</p>
            </div>
          ) : (
            <div className="courses-grid-modern">
              {courses.map((course, index) => (
                <motion.div
                  key={course.id}
                  className="course-module"
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: 0.6 + index * 0.1 }}
                  whileHover={{ scale: 1.03, z: 10 }}
                >
                  <div className="module-header">
                    <div className="module-badge">{course.subject}</div>
                    <div className="module-progress-badge">{course.progress}%</div>
                  </div>
                  <div className="module-body">
                    <div className="module-instructor">
                      <User size={16} strokeWidth={2} />
                      <span>{course.instructor}</span>
                    </div>
                    <div className="module-progress-track">
                      <motion.div 
                        className="module-progress-fill"
                        initial={{ width: 0 }}
                        animate={{ width: `${course.progress}%` }}
                        transition={{ duration: 0.8, delay: 0.7 + index * 0.1 }}
                      />
                    </div>
                  </div>
                  {course.testResults.length > 0 && (
                    <div className="module-tests">
                      <div className="tests-header-modern">
                        <BarChart3 size={16} strokeWidth={2} />
                        <span>Результаты</span>
                      </div>
                      <div className="tests-list-modern">
                        {course.testResults.map((test, i) => (
                          <div key={i} className="test-module">
                            <div className="test-name-modern">{test.name}</div>
                            <div className="test-score-modern">
                              <span className="score-highlight">{test.score}</span>
                              <span className="score-divider">/</span>
                              <span>{test.maxScore}</span>
                            </div>
                          </div>
                        ))}
                      </div>
                    </div>
                  )}
                </motion.div>
              ))}
            </div>
          )}
        </motion.div>

        <motion.div 
          className="insights-widget"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ delay: 0.8 }}
        >
          <div className="widget-section-header">
            <h2 className="section-title-modern">Аналитика</h2>
          </div>
          <div className="insights-grid">
            <div className="insight-card">
              <div className="insight-icon">
                <Clock size={20} strokeWidth={2} />
              </div>
              <div className="insight-content">
                <div className="insight-value">24</div>
                <div className="insight-label">Часов изучено</div>
              </div>
            </div>
            <div className="insight-card">
              <div className="insight-icon">
                <Calendar size={20} strokeWidth={2} />
              </div>
              <div className="insight-content">
                <div className="insight-value">12</div>
                <div className="insight-label">Дней подряд</div>
              </div>
            </div>
            <div className="insight-card">
              <div className="insight-icon">
                <Zap size={20} strokeWidth={2} />
              </div>
              <div className="insight-content">
                <div className="insight-value">+15%</div>
                <div className="insight-label">За неделю</div>
              </div>
            </div>
          </div>
        </motion.div>
      </div>
    </div>
  )
}

export default MyCourses
